import shutil
import json
import glob
import sys
import os
import re

# Get datapack directory, and adjust based on the filepath given
if len(sys.argv) < 3:
    raise IndexError("Please provide a filepath for the datapack directory, followed by the name of the overlay directory where changes should be made.")

if sys.argv[1] == ".":
    DATAPACK_DIR = os.getcwd()
elif sys.argv[1][0] in ["/", "~"]:
    DATAPACK_DIR = os.path.join(os.getcwd(), sys.argv[1])
else:
    DATAPACK_DIR = sys.argv[1]

OVERLAY_1_20_5 = os.path.join(DATAPACK_DIR, sys.argv[2])
TEMP_OVERLAY_DIR = os.path.join(DATAPACK_DIR, "TEMPOVERLAY")
REGEX_PATTERN = re.compile(r'\"value\"\s*:\s*\{([\w\s\d.\"-_]*\"min_inclusive\"\s*:\s-?\d+(.\d+)?\s*,[\w\s\d.\"-_]*\"max_(in|ex)clusive\"\s*:\s*-?\d+(.\d+)?[\w\s\d.\"-_]*)\s*\}\s*(,*)')
AFFECTED_DIRS = [
                    "data/minecraft/worldgen/placed_feature",
                    "data/minecraft/worldgen/configured_feature",
                    "data/minecraft/worldgen/configured_carver"
                ]

parsed_files = []


def read_mcmeta() -> dict:
    mcmeta_filepath = os.path.join(DATAPACK_DIR, "pack.mcmeta")

    if not os.path.exists(mcmeta_filepath):
        raise FileNotFoundError("Could not find the pack.mcmeta in the given datapack directory.")
    
    with open(mcmeta_filepath, 'r') as f:
        mcmeta_data = json.load(f)

    return mcmeta_data

def get_overlay_dirs() -> list[str]:
    mcmeta_data = read_mcmeta()

    for i, item in enumerate(mcmeta_data["overlays"]["entries"]):
        if isinstance(item["formats"], list):
            mcmeta_data["overlays"]["entries"][i]["formats"] = item["formats"][0]

    sorted_overlays = [
        overlay["directory"] for overlay in sorted(
                mcmeta_data["overlays"]["entries"],
                key= lambda d: d['formats'],
                reverse= True
            )
        ]
    sorted_overlays.append(".")
    
    return sorted_overlays


def main() -> None:
    # Get data and create temp folder
    overlays = get_overlay_dirs()

    if not os.path.exists(TEMP_OVERLAY_DIR):
        os.mkdir(TEMP_OVERLAY_DIR)

    # Loop through all the overlays and files
    for overlay in overlays:
        for affected_dir in AFFECTED_DIRS:
            for filename in glob.iglob(f"{os.path.join(DATAPACK_DIR, overlay, affected_dir)}" + '/**/*.json', recursive=True):
                local_filepath = filename.split(overlay, maxsplit=1)[1][1:]

                # Were we already were here?
                if local_filepath in parsed_files:
                    continue

                with open(filename, 'r') as f:
                    data = f.read()

                # Check if the int provider is present
                if not re.search(REGEX_PATTERN, data):
                    continue
                
                # We were already here!
                parsed_files.append(local_filepath)
                
                # Fix the encased int provider
                new_data = re.sub(REGEX_PATTERN, r"\g<1>\g<5>", data)
                new_json = json.loads(new_data)

                # Create the temp files
                temp_filepath = os.path.join(TEMP_OVERLAY_DIR, local_filepath)
                temp_dirpath = "/".join(temp_filepath.split("/")[:-1])
                if not os.path.exists(temp_dirpath):
                    os.makedirs(temp_dirpath)

                with open(temp_filepath, 'w') as f:
                    json.dump(new_json, f, indent=4)

    # Commit changes to 1.20.5 overlay
    for filepath in parsed_files:
        os.makedirs(os.path.dirname(os.path.join(OVERLAY_1_20_5, filepath)), exist_ok=True)
        shutil.copyfile(os.path.join(TEMP_OVERLAY_DIR, filepath), os.path.join(OVERLAY_1_20_5, filepath))

    # Clean up
    shutil.rmtree(TEMP_OVERLAY_DIR, ignore_errors=True)


if __name__ == '__main__':
    main()